
"""
Q1 Iteration-4 — Prime-width bands + narrow jitter (V2-compliant)
-----------------------------------------------------------------
This script reproduces the Iteration-4 gravitational-interferometer simulation
using only boolean/ordinal structural gates and PF/Born only at ties.

Key ideas:
- Geometry: vertical bands (structural ParentGate) on a 2-D grid of cells.
- Each cell requires an integer number of micro-acts. Deeper bands add small
  INTEGER increments per cell (no curve weights in acceptance).
- Time is act_count / M (M = micro-acts per tick). No background metric.
- Recombiner uses two boolean modular windows; probabilities arise from
  trial counts + PF/Born only for exact ties.

Defaults (Iteration-4 core slice):
- Lx=509 (prime), Ly=481, screen_col=508
- Bands increments added to base M: {+5s, +9s, +13s, +17s}
- Jitter half-width J = 0.15 * Theta; co-fit epsilon = 0.10 * Theta
- M=24, seeds={101,202,303}, s in {0,1,2,3}, Δh in {10,15}, y0=-15
- Slant OFF by default (set slant_list=[0,1] to also run ON)

Outputs (CSV):
- q1_struct_port_bias.csv
- q1_struct_peak_offsets.csv
- q1_struct_deltaT_unwrapped.csv
- q1_struct_manifest.json
- q1_struct_audit.json
"""

from __future__ import annotations
import numpy as np
import pandas as pd
from pathlib import Path
import json
from typing import Dict, Tuple

# ----------------------- CONFIG -----------------------
CONFIG: Dict = {
    "Lx": 509, "Ly": 481,
    "screen_col": 508,
    "Theta": 1.0,
    "eps_time": 0.10,          # co-fit half-width (ticks)
    "jitter_half_width": 0.15, # J * Theta
    "trials_per_condition": 50_000,
    "seeds": [101, 202, 303],
    "strictness_list": [0, 1, 2, 3],
    "delta_h_list": [10, 15],
    "y0_list": [-15],
    "M_list": [24],
    # Vertical bands & increments (INTS) added to base M, co-prime-ish w.r.t M.
    "band_increments": [0, 5, 9, 13, 17],  # band0..band4
    # Band edges in world-y (centered at 0). Negative is deeper.
    # Bands: [0,-20], [-20,-40], [-40,-60], [-60,-80], [-80,-120]
    "band_edges": [0, -20, -40, -60, -80, -120],
    # Slant: compute band from y - alpha*x instead of y
    "slant_list": [0],      # set to [0,1] to run both OFF and ON
    "slant_alpha": 1.0/19.0,
    # Peak-scan (pre-centered)
    "scan_half_width": 0.25, "scan_n_steps": 251, "trials_per_scan": 3_000,
    # Output files
    "out_port_bias": "q1_struct_port_bias.csv",
    "out_peaks": "q1_struct_peak_offsets.csv",
    "out_deltaT": "q1_struct_deltaT_unwrapped.csv",
    "out_manifest": "q1_struct_manifest.json",
    "out_audit": "q1_struct_audit.json",
}

# ----------------------- HELPERS -----------------------
def world_y_to_row(y_world: float, Ly: int) -> int:
    """Map world y in [-Ly//2, +Ly//2] to row index [0..Ly-1]; clamp at edges."""
    row = int(round(y_world + (Ly//2)))
    return max(0, min(Ly-1, row))

def fraction_part(x: float) -> float:
    """Return frac(x) in [0,1)."""
    f = x - np.floor(x)
    # guard against numerical negatives like -1e-16
    if f < 0: f += 1.0
    return float(f)

def resid_mod_Theta(x: np.ndarray, Theta: float) -> np.ndarray:
    """Distance to nearest integer multiple of Theta (here Theta=1)."""
    n = np.round(x / Theta)
    return np.abs(x - n * Theta)

def band_index(y_eff: np.ndarray, edges) -> np.ndarray:
    """
    Map effective y (centered at 0) to band indices 0..4 using edges:
      [0,-20], [-20,-40], [-40,-60], [-60,-80], [-80,-120]
    Anything deeper than -120 stays in band 4; anything >0 is band 0.
    """
    b = np.zeros_like(y_eff, dtype=np.int32)
    b[(y_eff < 0) & (y_eff >= -20)] = 1
    b[(y_eff < -20) & (y_eff >= -40)] = 2
    b[(y_eff < -40) & (y_eff >= -60)] = 3
    b[(y_eff < -60) & (y_eff >= -80)] = 4
    # y < -80 → band 4 (same increment)
    b[(y_eff < -80)] = 4
    return b

def build_cost_grid_vertical(Lx:int, Ly:int, s:int, M:int,
                             increments, slant:int, alpha:float) -> np.ndarray:
    """
    Build integer micro-act cost grid. Each cell:
      cost = M + s * inc[band(y_eff)]
    where y_eff = (row - Ly//2) - alpha*x if slant else (row - Ly//2).
    """
    yy, xx = np.mgrid[0:Ly, 0:Lx]
    y_world = yy - (Ly//2)
    if slant:
        y_eff = y_world - alpha * xx
    else:
        y_eff = y_world
    b = band_index(y_eff, CONFIG["band_edges"])
    inc_array = np.take(np.array(increments, dtype=np.int32), b)
    cost = (M + s * inc_array).astype(np.int32)
    return cost

def recombiner_stats(delta_t_ticks: float, Theta: float, eps_time: float,
                     K: int, seed: int, jitter_half_width: float) -> Dict:
    """
    Compute p0/p1/bias/tie/neutral rates for δt=0 at given ΔT using
    jitter uniform in [-J*Theta, +J*Theta], J = jitter_half_width.
    """
    rng = np.random.default_rng(seed)
    jitter = rng.uniform(-jitter_half_width*Theta, jitter_half_width*Theta, size=(K,))
    # fixed δt = 0 for bias-at-fixed-offset readout
    x0 = delta_t_ticks + jitter
    x1 = delta_t_ticks + jitter - 0.5*Theta
    co0 = (resid_mod_Theta(x0, Theta) <= eps_time)
    co1 = (resid_mod_Theta(x1, Theta) <= eps_time)
    both = co0 & co1
    only0 = co0 & ~co1
    only1 = ~co0 & co1
    neither = ~co0 & ~co1
    N = float(K)
    # PF/Born only at ties (both); neutral (neither) split 50/50 by the coupler (diagnostic).
    p0 = (only0.sum() + 0.5*both.sum() + 0.5*neither.sum()) / N
    p1 = (only1.sum() + 0.5*both.sum() + 0.5*neither.sum()) / N
    return {
        "p0": float(p0),
        "p1": float(p1),
        "bias": float(p0 - p1),
        "tie_rate_port0": float((only0 | both).mean()),
        "tie_rate_port1": float((only1 | both).mean()),
        "both_ports_rate": float(both.mean()),
        "neutral_rate": float(neither.mean()),
    }

def scan_peak(delta_t_ticks: float, Theta: float, eps_time: float,
              K_scan: int, seed: int, n_steps:int, half_width:float,
              jitter_half_width: float) -> Tuple[float, float]:
    """
    Pre-center scan:
      dT_mod = ((ΔT + 0.5) % 1) - 0.5
      dT_center = -dT_mod
      scan δt in [dT_center - 0.25, dT_center + 0.25] with 251 steps.
    """
    dT_mod = ((delta_t_ticks + 0.5) % 1.0) - 0.5
    dT_center = -dT_mod
    offsets = np.linspace(dT_center - half_width, dT_center + half_width, n_steps)
    rng = np.random.default_rng(seed)
    jitter = rng.uniform(-jitter_half_width*Theta, jitter_half_width*Theta, size=(K_scan,))
    p0_vals = []
    for off in offsets:
        x0 = delta_t_ticks + off + jitter
        x1 = delta_t_ticks + off + jitter - 0.5*Theta
        co0 = (resid_mod_Theta(x0, Theta) <= eps_time)
        co1 = (resid_mod_Theta(x1, Theta) <= eps_time)
        both = co0 & co1
        only0 = co0 & ~co1
        neither = ~co0 & ~co1
        N = float(K_scan)
        p0 = (only0.sum() + 0.5*both.sum() + 0.5*neither.sum()) / N
        p0_vals.append(p0)
    p0_vals = np.array(p0_vals)
    i = int(p0_vals.argmax())
    return float(offsets[i]), float(p0_vals[i])

# ----------------------- MAIN -----------------------
def run_iter4():
    C = CONFIG
    Lx, Ly = C["Lx"], C["Ly"]
    screen_col = C["screen_col"]
    Theta = C["Theta"]
    eps = C["eps_time"]
    K = C["trials_per_condition"]
    seeds = C["seeds"]
    strictness_list = C["strictness_list"]
    delta_h_list = C["delta_h_list"]
    y0_list = C["y0_list"]
    M_list = C["M_list"]
    increments = C["band_increments"]
    slant_list = C["slant_list"]
    alpha = C["slant_alpha"]
    J = C["jitter_half_width"]
    n_steps = C["scan_n_steps"]
    half_width = C["scan_half_width"]
    K_scan = C["trials_per_scan"]

    rows_bias = []
    rows_peaks = []
    rows_deltaT = []

    out_dir = Path(".")
    for M in M_list:
        for slant in slant_list:
            for s in strictness_list:
                # Build cost grid once per (M,slant,s)
                cost = build_cost_grid_vertical(Lx, Ly, s, M, increments, slant, alpha)
                row_cost = cost[:, :screen_col+1].sum(axis=1).astype(np.int64)
                for y0 in y0_list:
                    for dh in delta_h_list:
                        y_up = world_y_to_row(y0 + dh/2.0, Ly)
                        y_lo = world_y_to_row(y0 - dh/2.0, Ly)
                        # Time (ticks) = acts / M
                        tU = row_cost[y_up] / M
                        tL = row_cost[y_lo] / M
                        dT = tU - tL  # ticks (unwrapped)
                        frac_dT = fraction_part(dT)
                        rows_deltaT.append({
                            "slant": slant, "strictness": s, "dh": dh, "y0": y0, "M": M,
                            "raw_delta_t_ticks": float(dT), "frac_delta_t": frac_dT
                        })
                        # Bias at fixed δt=0 for each seed
                        for seed in seeds:
                            stats = recombiner_stats(dT, Theta, eps, K, seed, J)
                            rows_bias.append({
                                "slant": slant, "strictness": s, "dh": dh, "y0": y0, "M": M,
                                "seed": seed, "p0": stats["p0"], "p1": stats["p1"],
                                "bias": stats["bias"], "neutral_rate": stats["neutral_rate"],
                                "tie_rate_port0": stats["tie_rate_port0"],
                                "tie_rate_port1": stats["tie_rate_port1"],
                                "both_ports_rate": stats["both_ports_rate"],
                                "raw_delta_t_ticks": float(dT), "frac_delta_t": frac_dT
                            })
                            # Peak scan per condition with first seed
                            if seed == seeds[0]:
                                peak_off, peak_p0 = scan_peak(dT, Theta, eps, K_scan, seed, n_steps, half_width, J)
                                rows_peaks.append({
                                    "slant": slant, "strictness": s, "dh": dh, "y0": y0, "M": M, "seed": seed,
                                    "peak_delta_t_offset": peak_off, "peak_p0": peak_p0
                                })

    pd.DataFrame(rows_bias).to_csv(out_dir / C["out_port_bias"], index=False)
    pd.DataFrame(rows_peaks).to_csv(out_dir / C["out_peaks"], index=False)
    pd.DataFrame(rows_deltaT).to_csv(out_dir / C["out_deltaT"], index=False)

    # Minimal manifest and audit (JSON-only stdlib)
    with open(out_dir / C["out_manifest"], "w") as f:
        json.dump(C, f, indent=2)

    audit = {
        "guardrails": {
            "boolean_ordinal_acceptance": True,
            "pf_born_only_at_ties": True,
            "no_curve_weights_in_acceptance": True,
            "no_skip_neighbor_composition": True
        },
        "notes": "Structural ParentGate implemented as integer micro-acts; time (ticks) = micro_acts/M. Narrow jitter J=0.15, epsilon=0.10."
    }
    with open(out_dir / C["out_audit"], "w") as f:
        json.dump(audit, f, indent=2)

    print("Wrote:",
          C["out_port_bias"], C["out_peaks"], C["out_deltaT"],
          C["out_manifest"], C["out_audit"])

if __name__ == "__main__":
    run_iter4()
